# ZAR-Naira Exchange Backend API

A robust Node.js backend API for the ZAR-Naira Exchange platform with LowDB JSON database, JWT authentication, and password management.

## Features

- ✅ **Authentication System** with JWT tokens
- ✅ **LowDB JSON Database** for data persistence
- ✅ **Password Change Functionality**
- ✅ **Exchange Request Management**
- ✅ **Exchange Rates Management**
- ✅ **Admin Panel Support**
- ✅ **Secure Password Hashing** with bcrypt
- ✅ **Input Validation**
- ✅ **CORS Support**

## Prerequisites

- Node.js (v14 or higher)
- npm or yarn

## Installation

1. Navigate to the backend directory:
```bash
cd backend
```

2. Install dependencies:
```bash
npm install
```

3. Configure environment variables (`.env` file is already created with defaults)

4. Initialize the database and create admin user:
```bash
npm run init-db
```

## Default Admin Credentials

After initialization, you can login with:
- **Username:** `admin`
- **Password:** `admin123`

⚠️ **Important:** Change the admin password after first login!

## Running the Server

### Development Mode (with auto-restart)
```bash
npm run dev
```

### Production Mode
```bash
npm start
```

The server will start on `http://localhost:3000`

## API Endpoints

### Authentication

#### Login
```http
POST /api/auth/login
Content-Type: application/json

{
  "username": "admin",
  "password": "admin123"
}
```

#### Change Password
```http
POST /api/auth/change-password
Authorization: Bearer <token>
Content-Type: application/json

{
  "currentPassword": "admin123",
  "newPassword": "newSecurePassword123",
  "confirmPassword": "newSecurePassword123"
}
```

#### Verify Token
```http
GET /api/auth/verify
Authorization: Bearer <token>
```

#### Get Current User
```http
GET /api/auth/me
Authorization: Bearer <token>
```

### Exchange Operations

#### Get Exchange Rates (Public)
```http
GET /api/exchange/rates
```

#### Submit Exchange Request (Public)
```http
POST /api/exchange/request
Content-Type: application/json

{
  "name": "John Doe",
  "email": "john@example.com",
  "phone": "+27123456789",
  "amount": 1000,
  "currencyFrom": "ZAR",
  "currencyTo": "NGN"
}
```

#### Get All Exchange Requests (Admin Only)
```http
GET /api/exchange/requests?page=1&limit=10&status=pending
Authorization: Bearer <token>
```

#### Update Exchange Request Status (Admin Only)
```http
PATCH /api/exchange/requests/:id
Authorization: Bearer <token>
Content-Type: application/json

{
  "status": "completed"
}
```

#### Update Exchange Rates (Admin Only)
```http
PUT /api/exchange/rates/ZAR-NGN
Authorization: Bearer <token>
Content-Type: application/json

{
  "buyRate": 22.50,
  "sellRate": 23.00
}
```

### Health Check
```http
GET /api/health
```

## Database Schema

### Users Table
- `id` - Primary key
- `username` - Unique username
- `password` - Hashed password
- `email` - User email
- `role` - User role (admin)
- `created_at` - Creation timestamp
- `updated_at` - Last update timestamp

### Exchange Requests Table
- `id` - Primary key
- `name` - Customer name
- `email` - Customer email
- `phone` - Customer phone
- `amount` - Exchange amount
- `currency_from` - Source currency
- `currency_to` - Target currency
- `status` - Request status (pending, processing, completed, cancelled)
- `created_at` - Creation timestamp
- `updated_at` - Last update timestamp

### Exchange Rates Table
- `id` - Primary key
- `currency_pair` - Currency pair (e.g., ZAR-NGN)
- `buy_rate` - Buy rate
- `sell_rate` - Sell rate
- `updated_at` - Last update timestamp

## Security Features

- Password hashing with bcrypt (10 rounds)
- JWT token authentication
- Token expiration (24 hours)
- Input validation with express-validator
- Protected admin routes
- CORS configuration

## Environment Variables

```env
PORT=3000
JWT_SECRET=your-secret-key-change-this-in-production-12345
NODE_ENV=development
DATABASE_PATH=./database/exchange.db
```

⚠️ **Important:** Change the `JWT_SECRET` in production!

## Project Structure

```
backend/
├── config/
│   └── database.js          # Database configuration
├── middleware/
│   └── auth.js              # Authentication middleware
├── routes/
│   ├── auth.js              # Authentication routes
│   └── exchange.js          # Exchange routes
├── scripts/
│   └── init-db.js           # Database initialization script
├── database/
│   └── exchange.json        # LowDB JSON database (created after init)
├── .env                     # Environment variables
├── .gitignore
├── package.json
├── server.js                # Main server file
└── README.md
```

## Error Handling

The API returns standardized error responses:

```json
{
  "success": false,
  "message": "Error description",
  "errors": [] // Optional validation errors
}
```

## Success Responses

```json
{
  "success": true,
  "message": "Operation successful",
  "data": {} // Optional response data
}
```

## Development Notes

- The database file is automatically created on first run
- Admin user is created automatically during initialization
- All timestamps use ISO format
- Passwords are never returned in API responses
- Token expiration is set to 24 hours

## Testing the API

You can test the API using:
- Postman
- cURL
- Thunder Client (VS Code extension)
- Your frontend application

### Example cURL Request:

```bash
# Login
curl -X POST http://localhost:3000/api/auth/login \
  -H "Content-Type: application/json" \
  -d '{"username":"admin","password":"admin123"}'

# Change Password
curl -X POST http://localhost:3000/api/auth/change-password \
  -H "Content-Type: application/json" \
  -H "Authorization: Bearer YOUR_TOKEN" \
  -d '{
    "currentPassword":"admin123",
    "newPassword":"newPassword123",
    "confirmPassword":"newPassword123"
  }'
```

## Troubleshooting

### Database locked error
- Close any other connections to the database
- Restart the server

### JWT errors
- Check if token is properly formatted: `Bearer <token>`
- Verify token hasn't expired
- Ensure JWT_SECRET matches between sessions

### Port already in use
- Change PORT in .env file
- Kill process using port 3000: `netstat -ano | findstr :3000` (Windows)

## License

ISC

## Support

For issues and questions, please create an issue in the repository.
